const express = require("express");
const path = require("path");

const app = express();
let PORT = parseInt(process.env.PORT, 10) || 3000;
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static(path.join(__dirname, "public")));
let tasks = [];
let nextId = 1;
function validateTask(data) {
  const errors = [];

  if (!data.title || data.title.trim() === "") {
    errors.push("Title is required.");
  }

  if (!data.description || data.description.trim() === "") {
    errors.push("Description is required.");
  }

  if (!data.dueDate || data.dueDate.trim() === "") {
    errors.push("Due date is required.");
} else {
  const date = new Date(data.dueDate);
    if (isNaN(date.getTime())) {
      errors.push("Due date must be a valid date (YYYY-MM-DD).");
    }
  }
  const allowedStatuses = ["pending", "completed"];
  if (data.status && !allowedStatuses.includes(data.status)) {
    errors.push("Status must be either 'pending' or 'completed'.");
  }

  return errors;
}
app.get("/tasks", (req, res) => {
  res.json({
    success: true,
    data: tasks,
  });
});

app.get("/tasks/:id", (req, res) => {
  const id = parseInt(req.params.id, 10);
  const task = tasks.find((t) => t.id === id);

  if (!task) {
    return res.status(404).json({
      success: false,
      message: "Task not found.",
    });
  }

  res.json({
    success: true,
    data: task,
  });
});

app.post("/tasks", (req, res) => {
  const { title, description, status, dueDate } = req.body;

  const errors = validateTask({ title, description, status, dueDate });
  if (errors.length > 0) {
    return res.status(400).json({
      success: false,
      errors,
    });
  }

  const task = {
    id: nextId++,
    title: title.trim(),
    description: description.trim(),
    status: status || "pending",
    dueDate,
  };

  tasks.push(task);

  res.status(201).json({
    success: true,
    message: "Task created successfully.",
    data: task,
  });
});

app.put("/tasks/:id", (req, res) => {
  const id = parseInt(req.params.id, 10);
  const existingTask = tasks.find((t) => t.id === id);

  if (!existingTask) {
    return res.status(404).json({
      success: false,
      message: "Task not found.",
    });
  }

  const { title, description, status, dueDate } = req.body;

  const errors = validateTask({ title, description, status, dueDate });
  if (errors.length > 0) {
    return res.status(400).json({
      success: false,
      errors,
    });
  }

  existingTask.title = title.trim();
  existingTask.description = description.trim();
  existingTask.status = status || "pending";
  existingTask.dueDate = dueDate;

  res.json({
    success: true,
    message: "Task updated successfully.",
    data: existingTask,
  });
});

app.delete("/tasks/:id", (req, res) => {
  const id = parseInt(req.params.id, 10);
  const index = tasks.findIndex((t) => t.id === id);

  if (index === -1) {
    return res.status(404).json({
      success: false,
      message: "Task not found.",
    });
  }

  tasks.splice(index, 1);

  res.json({
    success: true,
    message: "Task deleted successfully.",
  });
});

const MAX_PORT_RETRIES = 10;

function startServer(port, attempt = 0) {
  const server = app.listen(port, () => {
    console.log(`Task Manager app running on http://localhost:${port}`);
  });

  server.on("error", (err) => {
    if (err && err.code === "EADDRINUSE") {
      if (attempt < MAX_PORT_RETRIES) {
        const nextPort = port + 1;
        console.warn(
          `Port ${port} is in use, trying available port ${nextPort} (attempt ${attempt + 1})...`
        );
        startServer(nextPort, attempt + 1);
      } else {
        console.error(
          `All ports from ${PORT} to ${port} are in use. Exiting.`
        );
        process.exit(1);
      }
    } else {
      console.error("Server error:", err);
      process.exit(1);
    }
  });
}

startServer(PORT);
